<?php
/*
Plugin Name: Disable All User Registrations
Description: Hard-disable user registrations from every common vector (core, REST API, WooCommerce, XML-RPC, multisite). Keeps "Anyone can register" off and blocks /wp-login.php?action=register.
Version: 1.0.0
Author: ChatGPT
License: GPLv2 or later
*/

if (!defined('ABSPATH')) { exit; }

/**
 * Force Membership off at the option level.
 * Also fix it at runtime even if something tries to flip it.
 */
register_activation_hook(__FILE__, function () {
    update_option('users_can_register', 0);
    if (is_multisite()) {
        // For multisite, disable network registration entirely.
        update_site_option('registration', 'none');
    }
});

// Always report "users cannot register".
add_filter('option_users_can_register', function ($value) { return false; }, 999, 1);
add_filter('pre_option_users_can_register', function ($value) { return 0; }, 999, 1);

// Multisite: force "no registration".
add_filter('pre_site_option_registration', function ($value) { return 'none'; }, 999, 1);

/**
 * Block access to /wp-login.php?action=register (redirect away).
 */
add_action('login_form_register', function () {
    // Allow admins creating users via wp-admin only; this runs on wp-login register screen.
    wp_safe_redirect(home_url('/'));
    exit;
});

// Remove default "Register" URLs if any theme/plugin calls them.
add_filter('register', '__return_empty_string', 999); // old themes sometimes echo register link
add_filter('wp_registration_url', function ($url) { return ''; }, 999);

/**
 * Add a hard error to any attempt that still makes it to the core registration handler.
 */
add_filter('registration_errors', function ($errors, $sanitized_user_login, $user_email) {
    if (!current_user_can('create_users')) {
        $errors->add('registration_closed', __('User registration is closed on this site.', 'disable-all-user-registrations'));
    }
    return $errors;
}, 999, 3);

/**
 * Disable XML-RPC entirely (some sites allow user creation via extensions).
 * If you rely on XML-RPC for something else, comment this out.
 */
add_filter('xmlrpc_enabled', '__return_false', 999);

/**
 * Remove REST API user endpoints to prevent programmatic creation.
 */
add_filter('rest_endpoints', function ($endpoints) {
    // Hard remove user collection and single endpoints
    if (isset($endpoints['/wp/v2/users'])) {
        unset($endpoints['/wp/v2/users']);
    }
    if (isset($endpoints['/wp/v2/users/(?P<id>[\d]+)'])) {
        unset($endpoints['/wp/v2/users/(?P<id>[\d]+)']);
    }
    if (isset($endpoints['/wp/v2/users/me'])) {
        unset($endpoints['/wp/v2/users/me']);
    }
    return $endpoints;
}, 999);

/**
 * As a second layer, deny any POST to users endpoint if another plugin re-adds it.
 */
add_filter('rest_request_before_callbacks', function ($response, $handler, $request) {
    $route  = $request->get_route();
    $method = $request->get_method();
    if ($method === 'POST' && preg_match('#^/wp/v2/users(\b|/|\?)#', $route)) {
        // Allow only admins inside wp-admin to create via REST (rare), otherwise block.
        if (!current_user_can('create_users') || !is_admin()) {
            return new WP_Error('registration_disabled', __('User registration via REST API is disabled.', 'disable-all-user-registrations'), array('status' => 403));
        }
    }
    return $response;
}, 10, 3);

/**
 * WooCommerce: disable account creation on My Account and Checkout screens regardless of settings.
 * Covers common and legacy filters.
 */
add_filter('woocommerce_registration_enabled', '__return_false', 999); // My Account creation
add_filter('woocommerce_enable_myaccount_registration', '__return_false', 999); // Legacy
add_filter('woocommerce_enable_signup_and_login_from_checkout', '__return_false', 999); // Legacy
add_filter('woocommerce_checkout_registration_enabled', '__return_false', 999); // Checkout
add_filter('woocommerce_create_account_default_checked', '__return_false', 999);

/**
 * BuddyPress / bbPress (if present): prevent any open registration hooks.
 * We don't call their functions directly, but we can gate the generic 'user_register' flow.
 */
add_action('user_register', function ($user_id) {
    // If not created by an admin inside wp-admin, immediately block and clean up.
    if (!current_user_can('create_users') || !is_admin()) {
        // Safety: avoid deleting service/system users accidentally.
        if ($user_id && $user_id > 1) {
            require_once ABSPATH . 'wp-admin/includes/user.php';
            wp_delete_user($user_id);
        }
        wp_die(__('User registration is disabled on this site.', 'disable-all-user-registrations'), 403);
        exit;
    }
}, 1);

/**
 * Hide "Anyone can register" checkbox to prevent confusion (visual only; we already force it off).
 */
add_action('admin_head-options-general.php', function () {
    echo '<style>#users_can_register{display:none!important;} label[for="users_can_register"]{opacity:.5;}</style>';
});

/**
 * Security header: if someone still hits /wp-login.php?action=register, send 410 Gone (optional).
 */
add_action('init', function () {
    if (is_admin() && isset($_GET['action']) && $_GET['action'] === 'register' && strpos($_SERVER['REQUEST_URI'], 'wp-login.php') !== false) {
        status_header(410);
        exit;
    }
});
