<?php
/*
Plugin Name: External Featured Image URL (Simple, No Underscore Meta)
Description: Menggunakan URL eksternal sebagai featured image dengan logika Index & Single. Memakai meta key tanpa underscore di depan agar mudah dipakai WP All Import. Tetap kompatibel dengan meta lama yang pakai underscore.
Author: ChatGPT & User
Version: 1.4.1
*/

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class External_Featured_Image_URL_Simple_No_Underscore {

    // Meta key BARU (tanpa underscore, dipersingkat)
    const META_URL_MAIN_NEW   = 'eifu_main';
    const META_URL_SINGLE_NEW = 'eifu_single';
    const META_MODE_NEW       = 'eifu_mode';

    // Meta key LAMA (pakai underscore) untuk kompatibilitas
    const META_URL_MAIN_OLD   = '_eifu_url_main';
    const META_URL_SINGLE_OLD = '_eifu_url_single_alt';
    const META_MODE_OLD       = '_eifu_display';

    public function __construct() {
        add_action( 'add_meta_boxes', array( $this, 'register_meta_box' ) );
        add_action( 'save_post', array( $this, 'save_meta' ) );

        add_filter( 'post_thumbnail_html', array( $this, 'filter_post_thumbnail_html' ), 10, 5 );
        add_filter( 'get_post_metadata', array( $this, 'fake_thumbnail_id' ), 10, 4 );
    }

    /**
     * Register meta box di bawah editor.
     */
    public function register_meta_box() {
        $post_types = apply_filters( 'eifu_post_types', array( 'post' ) );

        foreach ( $post_types as $pt ) {
            add_meta_box(
                'eifu_meta_box',
                __( 'External Featured Image URL', 'external-featured-url-simple' ),
                array( $this, 'render_meta_box' ),
                $pt,
                'normal',
                'high'
            );
        }
    }

    /**
     * Ambil meta dengan prioritas: NEW lalu OLD.
     */
    protected function get_meta_with_fallback( $post_id, $new_key, $old_key ) {
        $value = get_post_meta( $post_id, $new_key, true );
        if ( '' === $value ) {
            $value = get_post_meta( $post_id, $old_key, true );
        }
        return $value;
    }

    protected function get_mode_setting( $post_id ) {
        $mode = $this->get_meta_with_fallback( $post_id, self::META_MODE_NEW, self::META_MODE_OLD );
        $allowed = array( 'both', 'index', 'single', 'index_diff_single' );
        if ( empty( $mode ) || ! in_array( $mode, $allowed, true ) ) {
            $mode = 'both';
        }
        return $mode;
    }

    /**
     * Tampilkan form meta box.
     */
    public function render_meta_box( $post ) {
        wp_nonce_field( 'eifu_save_meta', 'eifu_nonce' );

        $url_main   = $this->get_meta_with_fallback( $post->ID, self::META_URL_MAIN_NEW, self::META_URL_MAIN_OLD );
        $url_single = $this->get_meta_with_fallback( $post->ID, self::META_URL_SINGLE_NEW, self::META_URL_SINGLE_OLD );
        $mode       = $this->get_mode_setting( $post->ID );
        ?>
        <p>
            <strong><?php _e( 'Kolom Utama (URL Gambar Eksternal)', 'external-featured-url-simple' ); ?></strong><br>
            <input type="text" name="eifu_main" id="eifu_main"
                   value="<?php echo esc_attr( $url_main ); ?>"
                   style="width:100%; max-width:700px;"
                   placeholder="https://example.com/image.jpg">
            <br><small>Meta key: <code>eifu_main</code></small>
        </p>

        <p>
            <strong><?php _e( 'Pengaturan Tampilan', 'external-featured-url-simple' ); ?></strong><br>
            <select name="eifu_mode" id="eifu_mode">
                <option value="both" <?php selected( $mode, 'both' ); ?>>
                    <?php _e( 'Index & Single (default)', 'external-featured-url-simple' ); ?>
                </option>
                <option value="index" <?php selected( $mode, 'index' ); ?>>
                    <?php _e( 'Index saja', 'external-featured-url-simple' ); ?>
                </option>
                <option value="single" <?php selected( $mode, 'single' ); ?>>
                    <?php _e( 'Single saja', 'external-featured-url-simple' ); ?>
                </option>
                <option value="index_diff_single" <?php selected( $mode, 'index_diff_single' ); ?>>
                    <?php _e( 'Index beda Single', 'external-featured-url-simple' ); ?>
                </option>
            </select>
            <br><small>Meta key: <code>eifu_mode</code></small>
        </p>

        <div id="eifu_single_wrapper" style="<?php echo ( $mode === 'index_diff_single' ) ? '' : 'display:none;'; ?>">
            <p>
                <strong><?php _e( 'Kolom Single (URL untuk Single Post jika berbeda)', 'external-featured-url-simple' ); ?></strong><br>
                <input type="text" name="eifu_single" id="eifu_single"
                       value="<?php echo esc_attr( $url_single ); ?>"
                       style="width:100%; max-width:700px;"
                       placeholder="https://example.com/image-single.jpg">
                <br><small>Meta key: <code>eifu_single</code></small>
            </p>
        </div>

        <?php if ( $url_main || $url_single ) : ?>
            <hr>
            <style>
                .eifu-preview-box {
                    background: #fafafa;
                    padding: 10px 12px;
                    border: 1px solid #ddd;
                    border-radius: 6px;
                    margin-bottom: 15px;
                }
                .eifu-link {
                    display: block;
                    max-width: 100%;
                    word-wrap: break-word;
                    overflow-wrap: break-word;
                    white-space: normal;
                    margin-bottom: 8px;
                    font-size: 13px;
                }
            </style>
            <p><strong><?php _e( 'Preview / Link Gambar', 'external-featured-url-simple' ); ?></strong></p>
            <?php if ( $url_main ) : ?>
                <div class="eifu-preview-box">
                    <strong><?php _e( 'Index / Default:', 'external-featured-url-simple' ); ?></strong><br>
                    <a class="eifu-link" href="<?php echo esc_url( $url_main ); ?>" target="_blank" rel="noopener noreferrer">
                        <?php echo esc_html( $url_main ); ?>
                    </a>
                    <img src="<?php echo esc_url( $url_main ); ?>" alt="" style="max-width:100%; height:auto; margin-top:6px;">
                </div>
            <?php endif; ?>
            <?php if ( $url_single ) : ?>
                <div class="eifu-preview-box">
                    <strong><?php _e( 'Single Post:', 'external-featured-url-simple' ); ?></strong><br>
                    <a class="eifu-link" href="<?php echo esc_url( $url_single ); ?>" target="_blank" rel="noopener noreferrer">
                        <?php echo esc_html( $url_single ); ?>
                    </a>
                    <img src="<?php echo esc_url( $url_single ); ?>" alt="" style="max-width:100%; height:auto; margin-top:6px;">
                </div>
            <?php endif; ?>
<?php endif; ?>

        <script>
        document.addEventListener('DOMContentLoaded', function () {
            var sel  = document.getElementById('eifu_mode');
            var box  = document.getElementById('eifu_single_wrapper');

            function updateBox() {
                if (!sel || !box) return;
                box.style.display = sel.value === 'index_diff_single' ? '' : 'none';
            }

            updateBox();
            if (sel) {
                sel.addEventListener('change', updateBox);
            }
        });
        </script>
        <?php
    }

    /**
     * Simpan meta BARU (tanpa underscore). Meta lama tetap dibiarkan agar kompatibel,
     * tapi yang dipakai di front-end adalah meta baru jika ada.
     */
    public function save_meta( $post_id ) {
        if ( ! isset( $_POST['eifu_nonce'] ) || ! wp_verify_nonce( $_POST['eifu_nonce'], 'eifu_save_meta' ) ) {
            return;
        }

        if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) {
            return;
        }

        if ( ! current_user_can( 'edit_post', $post_id ) ) {
            return;
        }

        $url_main   = isset( $_POST['eifu_main'] )   ? esc_url_raw( trim( wp_unslash( $_POST['eifu_main'] ) ) )   : '';
        $url_single = isset( $_POST['eifu_single'] ) ? esc_url_raw( trim( wp_unslash( $_POST['eifu_single'] ) ) ) : '';
        $mode       = isset( $_POST['eifu_mode'] )   ? sanitize_key( wp_unslash( $_POST['eifu_mode'] ) )          : 'both';

        $allowed_mode = array( 'both', 'index', 'single', 'index_diff_single' );
        if ( ! in_array( $mode, $allowed_mode, true ) ) {
            $mode = 'both';
        }

        // Simpan meta baru
        if ( $url_main ) {
            update_post_meta( $post_id, self::META_URL_MAIN_NEW, $url_main );
        } else {
            delete_post_meta( $post_id, self::META_URL_MAIN_NEW );
        }

        if ( $url_single ) {
            update_post_meta( $post_id, self::META_URL_SINGLE_NEW, $url_single );
        } else {
            delete_post_meta( $post_id, self::META_URL_SINGLE_NEW );
        }

        update_post_meta( $post_id, self::META_MODE_NEW, $mode );
    }

    /**
     * Tentukan URL mana yang dipakai berdasarkan konteks (index / single) dan mode.
     */
    protected function get_url_for_current_context( $post_id ) {
        $post_id = (int) $post_id;

        $url_main   = $this->get_meta_with_fallback( $post_id, self::META_URL_MAIN_NEW, self::META_URL_MAIN_OLD );
        $url_single = $this->get_meta_with_fallback( $post_id, self::META_URL_SINGLE_NEW, self::META_URL_SINGLE_OLD );
        $mode       = $this->get_mode_setting( $post_id );

        if ( empty( $url_main ) && empty( $url_single ) ) {
            return '';
        }

        // Single post context
        if ( is_singular( 'post' ) && (int) get_queried_object_id() === $post_id ) {
            if ( 'single' === $mode || 'both' === $mode ) {
                return $url_main;
            }
            if ( 'index_diff_single' === $mode ) {
                return $url_single ? $url_single : $url_main;
            }
            return '';
        }

        // Index / archive / search context
        if ( is_home() || is_archive() || is_search() ) {
            if ( 'index' === $mode || 'both' === $mode || 'index_diff_single' === $mode ) {
                return $url_main;
            }
            return '';
        }

        // Default fallback
        return ( 'both' === $mode ) ? $url_main : '';
    }

    /**
     * Ganti HTML featured image dengan <img> URL eksternal.
     */
    public function filter_post_thumbnail_html( $html, $post_id, $post_thumbnail_id, $size, $attr ) {
        if ( is_admin() ) {
            return $html;
        }

        $url = $this->get_url_for_current_context( $post_id );

        if ( ! $url ) {
            return $html;
        }

        $attributes = array();
        if ( is_array( $attr ) ) {
            $attributes = $attr;
        }

        if ( empty( $attributes['alt'] ) ) {
            $attributes['alt'] = get_the_title( $post_id );
        }

        $attr_pairs = array();
        foreach ( $attributes as $key => $value ) {
            if ( '' === $value && 'alt' !== $key ) {
                continue;
            }
            $attr_pairs[] = sprintf( '%s="%s"', esc_attr( $key ), esc_attr( $value ) );
        }

        $attr_string = '';
        if ( ! empty( $attr_pairs ) ) {
            $attr_string = ' ' . implode( ' ', $attr_pairs );
        }

        return sprintf(
            '<img src="%s"%s />',
            esc_url( $url ),
            $attr_string
        );
    }

    /**
     * Palsukan _thumbnail_id kalau tidak ada tapi ada URL eksternal.
     */
    public function fake_thumbnail_id( $value, $post_id, $meta_key, $single ) {
        if ( '_thumbnail_id' !== $meta_key ) {
            return $value;
        }

        if ( ! empty( $value ) ) {
            return $value;
        }

        if ( is_admin() ) {
            return $value;
        }

        $url = $this->get_url_for_current_context( $post_id );
        if ( $url ) {
            return -1;
        }

        return $value;
    }
}

new External_Featured_Image_URL_Simple_No_Underscore();
