<?php
/**
 * Plugin Name: Indo Date Converter
 * Description: Konversi tanggal WordPress ke Bahasa Indonesia (hari/bulan panjang & singkat). Shortcode [tanggal], fungsi indo_date(), dan halaman Settings (hapus koma, singkatan Agustus, default format/TZ, kapitalisasi, ejaan "Jumat/Jum’at", custom mapping, paksa 24-jam, ganti pemisah tanggal).
 * Version:     1.3.0
 * Author:      Your Name
 * License:     GPL-2.0+
 * Text Domain: indo-date-converter
 */

if ( ! defined( 'ABSPATH' ) ) exit;

class Indo_Date_Converter {
    private static $instance = null;

    const OPT_KEY = 'indo_date_converter_options';

    private $opts = [
        'remove_comma_after_weekday' => 0,            // 0/1
        'short_aug_style'            => 'Agu',        // 'Agu' or 'Ags'
        'shortcode_default_format'   => 'l, d F Y',
        'shortcode_default_tz'       => '',           // empty -> WP timezone
        'capitalization'             => 'asis',       // asis|title|lower|upper
        'use_jumat_apostrophe'       => 0,            // 0 = "Jumat", 1 = "Jum’at"
        'custom_mapping'             => '',           // "Friday=Jum’at" per line
        // NEW in 1.3.0
        'force_24h'                  => 0,            // 0/1
        'date_separator'             => '',           // '', '-', '/', '.', 'space'
    ];

    private $map = [];
    private $ordinal_regex = '/\b(\d{1,2})(st|nd|rd|th)\b/';

    public static function instance() {
        return self::$instance ?: self::$instance = new self();
    }

    private function __construct() {
        $this->opts = wp_parse_args( get_option( self::OPT_KEY, [] ), $this->opts );
        $this->build_map();

        add_filter( 'date_i18n',        [ $this, 'convert' ], 999, 4 );
        add_filter( 'wp_date',          [ $this, 'convert' ], 999, 4 );
        add_filter( 'the_time',         [ $this, 'convert' ], 999, 2 );
        add_filter( 'the_date',         [ $this, 'convert' ], 999, 2 );
        add_filter( 'get_the_date',     [ $this, 'convert' ], 999, 3 );
        add_filter( 'get_the_time',     [ $this, 'convert' ], 999, 3 );
        add_filter( 'get_comment_date', [ $this, 'convert' ], 999, 3 );
        add_filter( 'get_comment_time', [ $this, 'convert' ], 999, 5 );

        add_shortcode( 'tanggal', [ $this, 'shortcode_tanggal' ] );

        if ( ! function_exists( 'indo_date' ) ) {
            function indo_date( $format = 'l, d F Y', $timestamp = null, $timezone = '' ) {
                return Indo_Date_Converter::instance()->format_indo( $format, $timestamp, $timezone );
            }
        }

        if ( is_admin() ) {
            add_action( 'admin_menu',  [ $this, 'admin_menu' ] );
            add_action( 'admin_init',  [ $this, 'register_settings' ] );
            add_filter( 'plugin_action_links_' . plugin_basename(__FILE__), [ $this, 'plugin_action_links' ] );
        }
    }

    private function build_map() {
        $aug_short   = ( $this->opts['short_aug_style'] === 'Ags' ) ? 'Ags' : 'Agu';
        $aug_short_l = strtolower( $aug_short );

        $map = [
            // Hari - panjang
            'Monday'    => 'Senin',
            'Tuesday'   => 'Selasa',
            'Wednesday' => 'Rabu',
            'Thursday'  => 'Kamis',
            'Friday'    => 'Jumat',
            'Saturday'  => 'Sabtu',
            'Sunday'    => 'Minggu',
            // Hari - singkat
            'Mon' => 'Sen', 'Tue' => 'Sel', 'Wed' => 'Rab', 'Thu' => 'Kam', 'Fri' => 'Jum', 'Sat' => 'Sab', 'Sun' => 'Min',
            // Bulan - panjang
            'January'   => 'Januari',
            'February'  => 'Februari',
            'March'     => 'Maret',
            'April'     => 'April',
            'May'       => 'Mei',
            'June'      => 'Juni',
            'July'      => 'Juli',
            'August'    => 'Agustus',
            'September' => 'September',
            'October'   => 'Oktober',
            'November'  => 'November',
            'December'  => 'Desember',
            // Bulan - singkat
            'Jan' => 'Jan', 'Feb' => 'Feb', 'Mar' => 'Mar', 'Apr' => 'Apr', 'May' => 'Mei',
            'Jun' => 'Jun', 'Jul' => 'Jul', 'Aug' => $aug_short, 'Sep' => 'Sep', 'Oct' => 'Okt', 'Nov' => 'Nov', 'Dec' => 'Des',
            // lower-case
            'monday'    => 'senin',   'tuesday'   => 'selasa', 'wednesday' => 'rabu',
            'thursday'  => 'kamis',   'friday'    => 'jumat',  'saturday'  => 'sabtu', 'sunday' => 'minggu',
            'mon' => 'sen', 'tue' => 'sel', 'wed' => 'rab', 'thu' => 'kam', 'fri' => 'jum', 'sat' => 'sab', 'sun' => 'min',
            'january'   => 'januari', 'february'  => 'februari', 'march'   => 'maret',   'april' => 'april',
            'may'       => 'mei',     'june'      => 'juni',     'july'    => 'juli',    'august' => 'agustus',
            'september' => 'september','october'  => 'oktober',  'november'=> 'november','december'=> 'desember',
            'jan' => 'jan', 'feb' => 'feb', 'mar' => 'mar', 'apr' => 'apr', 'may' => 'mei',
            'jun' => 'jun', 'jul' => 'jul', 'aug' => $aug_short_l, 'sep' => 'sep', 'oct' => 'okt', 'nov' => 'nov', 'dec' => 'des',
        ];

        if ( ! empty( $this->opts['use_jumat_apostrophe'] ) ) {
            $map['Friday'] = 'Jum’at';
            $map['friday'] = 'jum’at';
            $map['Fri']    = 'Jum';
            $map['fri']    = 'jum';
        }

        $custom = $this->parse_custom_mapping( $this->opts['custom_mapping'] ?? '' );
        if ( $custom ) {
            foreach ( $custom as $en => $id ) {
                $map[$en] = $id;
            }
        }

        $this->map = $map;
    }

    public function shortcode_tanggal( $atts ) {
        $atts = shortcode_atts( [
            'format'   => $this->opts['shortcode_default_format'],
            'waktu'    => 'now',
            'zona'     => $this->opts['shortcode_default_tz'],
        ], $atts, 'tanggal' );

        $ts = $this->parse_to_timestamp( $atts['waktu'], $atts['zona'] );
        return esc_html( $this->format_indo( $atts['format'], $ts, $atts['zona'] ) );
    }

    public function format_indo( $format = 'l, d F Y', $timestamp = null, $timezone = '' ) {
        if ( empty( $timestamp ) ) $timestamp = current_time( 'timestamp' );

        if ( empty( $timezone ) ) {
            $wp_tz = get_option( 'timezone_string' );
            $timezone = $wp_tz ? $wp_tz : 'UTC';
        }

        $formatted = function_exists('wp_date')
            ? wp_date( $format, $timestamp, new \DateTimeZone( $timezone ) )
            : date_i18n( $format, $timestamp, true );

        return $this->translate_string( $formatted );
    }

    public function convert( $date_string ) {
        $clean = preg_replace( $this->ordinal_regex, '$1', $date_string );
        return $this->translate_string( $clean );
    }

    private function translate_string( $str ) {
        $keys = array_keys( $this->map );
        usort( $keys, function( $a, $b ){ return strlen($b) - strlen($a); } );

        foreach ( $keys as $en ) {
            $id = $this->map[ $en ];
            $pattern = '/(?<=^|[\s,.-])' . preg_quote( $en, '/' ) . '(?=$|[\s,.-])/';
            $str = preg_replace( $pattern, $id, $str );
        }

        if ( ! empty( $this->opts['remove_comma_after_weekday'] ) ) {
            $str = preg_replace('/\b(Senin|Selasa|Rabu|Kamis|Jumat|Jum’at|Sabtu|Minggu),\s*/u', '$1 ', $str);
            $str = preg_replace('/\b(senin|selasa|rabu|kamis|jumat|jum’at|sabtu|minggu),\s*/u', '$1 ', $str);
        }

        if ( ! empty( $this->opts['force_24h'] ) ) {
            $str = $this->convert_time_12_to_24( $str );
        }

        if ( ! empty( $this->opts['date_separator'] ) ) {
            $sep = $this->opts['date_separator'] === 'space' ? ' ' : $this->opts['date_separator'];
            // Ganti pemisah antara angka tanggal-bulan-tahun (hindari jam "13:45")
            $str = preg_replace_callback(
                '/(\d{1,4})[\/\.\-](\d{1,2})[\/\.\-](\d{2,4})/u',
                function($m) use ($sep){ return $m[1].$sep.$m[2].$sep.$m[3]; },
                $str
            );
        }

        $str = $this->apply_capitalization( $str, $this->opts['capitalization'] );

        return $str;
    }

    /** Ubah 'h:mm am/pm' / 'hh:mm AM/PM' menjadi 24-jam */
    private function convert_time_12_to_24( $s ) {
        // 1) pola hh:mm am/pm
        $s = preg_replace_callback(
            '/\b(0?\d|1[0-2]):([0-5]\d)\s*([ap]m)\b/i',
            function($m){
                $h = intval($m[1]); $min = $m[2]; $ampm = strtolower($m[3]);
                if ($ampm === 'pm' && $h != 12) $h += 12;
                if ($ampm === 'am' && $h == 12) $h = 0;
                return sprintf('%02d:%s', $h, $min);
            }, $s
        );
        // 2) pola hh am/pm (tanpa menit)
        $s = preg_replace_callback(
            '/\b(0?\d|1[0-2])\s*([ap]m)\b/i',
            function($m){
                $h = intval($m[1]); $ampm = strtolower($m[2]);
                if ($ampm === 'pm' && $h != 12) $h += 12;
                if ($ampm === 'am' && $h == 12) $h = 0;
                return sprintf('%02d:00', $h);
            }, $s
        );
        // 3) hapus label AM/PM tersisa yang berdiri sendiri
        $s = preg_replace('/\b(AM|PM|am|pm)\b/', '', $s);
        return trim(preg_replace('/\s{2,}/', ' ', $s));
    }

    private function apply_capitalization( $s, $mode ) {
        switch ($mode) {
            case 'title':
                if ( function_exists('mb_convert_case') ) {
                    return mb_convert_case( $s, MB_CASE_TITLE, 'UTF-8' );
                }
                return ucwords( strtolower( $s ) );
            case 'lower':
                return function_exists('mb_strtolower') ? mb_strtolower($s,'UTF-8') : strtolower($s);
            case 'upper':
                return function_exists('mb_strtoupper') ? mb_strtoupper($s,'UTF-8') : strtoupper($s);
            case 'asis':
            default:
                return $s;
        }
    }

    private function parse_to_timestamp( $time, $timezone = '' ) {
        try {
            if ( is_numeric( $time ) ) return (int) $time;
            if ( strtolower( trim( $time ) ) === 'now' ) return current_time( 'timestamp' );
            $tz_name = $timezone ?: ( get_option( 'timezone_string' ) ?: 'UTC' );
            $tz = new \DateTimeZone( $tz_name );
            $dt = new \DateTime( $time, $tz );
            return $dt->getTimestamp();
        } catch ( \Exception $e ) {
            return current_time( 'timestamp' );
        }
    }

    private function parse_custom_mapping( $text ) {
        $out = [];
        $lines = preg_split( '/\r\n|\r|\n/', (string) $text );
        foreach ( $lines as $line ) {
            $line = trim( $line );
            if ( $line === '' || strpos( $line, '=' ) === false ) continue;
            list( $k, $v ) = array_map( 'trim', explode( '=', $line, 2 ) );
            if ( $k === '' ) continue;
            $out[$k] = $v;
        }
        return $out;
    }

    public function admin_menu() {
        add_options_page(
            __( 'Indo Date Settings', 'indo-date-converter' ),
            __( 'Indo Date', 'indo-date-converter' ),
            'manage_options',
            'indo-date-converter',
            [ $this, 'settings_page' ]
        );
    }

    public function register_settings() {
        register_setting(
            'indo_date_converter_group',
            self::OPT_KEY,
            [ 'type' => 'array', 'sanitize_callback' => [ $this, 'sanitize_options' ] ]
        );

        add_settings_section(
            'indo_date_main',
            __( 'Pengaturan Gaya Tanggal', 'indo-date-converter' ),
            function(){ echo '<p>Atur gaya tampilan penanggalan Bahasa Indonesia.</p>'; },
            'indo-date-converter'
        );

        $this->add_field_checkbox(
            'remove_comma_after_weekday',
            __( 'Hapus koma setelah nama hari', 'indo-date-converter' ),
            __( 'Contoh: "Senin, 01 Januari" → "Senin 01 Januari"', 'indo-date-converter' )
        );

        $this->add_field_select(
            'short_aug_style',
            __( 'Singkatan bulan Agustus', 'indo-date-converter' ),
            [ 'Agu' => 'Agu', 'Ags' => 'Ags' ],
            __( 'Pilih singkatan untuk "Aug".', 'indo-date-converter' )
        );

        $this->add_field_text(
            'shortcode_default_format',
            __( 'Default format shortcode', 'indo-date-converter' ),
            __( 'Dipakai saat [tanggal] tanpa atribut format. Contoh: l, d F Y H:i', 'indo-date-converter' ),
            'l, d F Y'
        );

        $this->add_field_text(
            'shortcode_default_tz',
            __( 'Default timezone shortcode', 'indo-date-converter' ),
            __( 'Kosongkan untuk mengikuti timezone WordPress (Settings → General). Contoh: Asia/Jakarta', 'indo-date-converter' ),
            'Asia/Jakarta'
        );

        add_settings_field(
            'capitalization',
            __( 'Kapitalisasi', 'indo-date-converter' ),
            [ $this, 'field_select' ],
            'indo-date-converter',
            'indo_date_main',
            [
                'key'   => 'capitalization',
                'items' => [
                    'asis'  => 'As-is (apa adanya)',
                    'title' => 'Title Case',
                    'lower' => 'lowercase',
                    'upper' => 'UPPERCASE',
                ],
                'desc'  => __( 'Aturan kapitalisasi diterapkan ke hasil akhir.', 'indo-date-converter' ),
            ]
        );

        $this->add_field_checkbox(
            'use_jumat_apostrophe',
            __( 'Gunakan ejaan "Jum’at" (dengan apostrof)', 'indo-date-converter' ),
            __( 'Jika tidak dicentang, gunakan "Jumat".', 'indo-date-converter' )
        );

        add_settings_field(
            'custom_mapping',
            __( 'Custom mapping (override)', 'indo-date-converter' ),
            [ $this, 'field_textarea' ],
            'indo-date-converter',
            'indo_date_main',
            [
                'key'  => 'custom_mapping',
                'desc' => __( 'Satu per baris, format: English=Indonesia. Contoh: Friday=Jum’at | Wed=Rabu | August=Agustus', 'indo-date-converter' ),
                'placeholder' => "Friday=Jum’at\nWed=Rabu\nAugust=Agustus",
            ]
        );

        // NEW in 1.3.0: Force 24h
        $this->add_field_checkbox(
            'force_24h',
            __( 'Paksa format waktu 24-jam', 'indo-date-converter' ),
            __( 'Mengubah tampilan 12-jam (am/pm) menjadi 24-jam pada keluaran akhir.', 'indo-date-converter' )
        );

        // NEW in 1.3.0: Date separator
        add_settings_field(
            'date_separator',
            __( 'Pemisah tanggal (DD/MM/YYYY)', 'indo-date-converter' ),
            [ $this, 'field_select' ],
            'indo-date-converter',
            'indo_date_main',
            [
                'key'   => 'date_separator',
                'items' => [
                    ''       => __( 'Biarkan sesuai sumber (default)', 'indo-date-converter' ),
                    '-'      => '-',
                    '/'      => '/',
                    '.'      => '.',
                    'space'  => __( 'Spasi', 'indo-date-converter' ),
                ],
                'desc'  => __( 'Menerapkan ke pola numerik tanggal umum (mis. 01/11/2025 atau 2025-11-01). Tidak memengaruhi pemisah waktu (:) ', 'indo-date-converter' ),
            ]
        );
    }

    public function sanitize_options( $input ) {
        $out = wp_parse_args( (array) $input, $this->opts );

        $out['remove_comma_after_weekday'] = empty( $input['remove_comma_after_weekday'] ) ? 0 : 1;

        $allowed_aug = [ 'Agu', 'Ags' ];
        $out['short_aug_style'] = in_array( $input['short_aug_style'] ?? 'Agu', $allowed_aug, true ) ? $input['short_aug_style'] : 'Agu';

        $out['shortcode_default_format'] = sanitize_text_field( $input['shortcode_default_format'] ?? 'l, d F Y' );
        $out['shortcode_default_tz']     = sanitize_text_field( trim( (string) ( $input['shortcode_default_tz'] ?? '' ) ) );

        $allowed_caps = [ 'asis', 'title', 'lower', 'upper' ];
        $cap = $input['capitalization'] ?? 'asis';
        $out['capitalization'] = in_array( $cap, $allowed_caps, true ) ? $cap : 'asis';

        $out['use_jumat_apostrophe'] = empty( $input['use_jumat_apostrophe'] ) ? 0 : 1;

        $txt = (string) ( $input['custom_mapping'] ?? '' );
        $out['custom_mapping'] = wp_kses_post( trim( $txt ) );

        $out['force_24h'] = empty( $input['force_24h'] ) ? 0 : 1;

        $allowed_sep = [ '', '-', '/', '.', 'space' ];
        $sep = $input['date_separator'] ?? '';
        $out['date_separator'] = in_array( $sep, $allowed_sep, true ) ? $sep : '';

        $this->opts = $out;
        $this->build_map();

        return $out;
    }

    public function settings_page() {
        ?>
        <div class="wrap">
            <h1><?php esc_html_e( 'Indo Date Settings', 'indo-date-converter' ); ?></h1>
            <form method="post" action="options.php">
                <?php
                    settings_fields( 'indo_date_converter_group' );
                    do_settings_sections( 'indo-date-converter' );
                    submit_button();
                ?>
            </form>
            <hr>
            <h2><?php esc_html_e('Preview', 'indo-date-converter'); ?></h2>
            <p><?php echo esc_html( $this->format_indo( $this->opts['shortcode_default_format'] ?: 'l, d F Y H:i' ) ); ?></p>
            <p class="description"><?php esc_html_e('Preview mengikuti pengaturan saat ini.', 'indo-date-converter'); ?></p>
            <hr>
            <h2><?php esc_html_e('Contoh Penggunaan', 'indo-date-converter'); ?></h2>
            <p><code>[tanggal]</code> • <code>[tanggal format="l, d F Y H:i" zona="Asia/Jakarta"]</code></p>
            <p><code>&lt;?php echo indo_date('l, d F Y'); ?&gt;</code></p>
        </div>
        <?php
    }

    private function add_field_checkbox( $key, $title, $label ) {
        add_settings_field(
            $key, $title, [ $this, 'field_checkbox' ], 'indo-date-converter', 'indo_date_main',
            [ 'key' => $key, 'label' => $label ]
        );
    }

    private function add_field_text( $key, $title, $desc = '', $placeholder = '' ) {
        add_settings_field(
            $key, $title, [ $this, 'field_text' ], 'indo-date-converter', 'indo_date_main',
            [ 'key' => $key, 'desc' => $desc, 'placeholder' => $placeholder ]
        );
    }

    private function add_field_select( $key, $title, $items, $desc = '' ) {
        add_settings_field(
            $key, $title, [ $this, 'field_select' ], 'indo-date-converter', 'indo_date_main',
            [ 'key' => $key, 'items' => $items, 'desc' => $desc ]
        );
    }

    public function field_checkbox( $args ) {
        $key   = $args['key'];
        $label = $args['label'] ?? '';
        $val   = ! empty( $this->opts[ $key ] ) ? 1 : 0;
        echo '<label><input type="checkbox" name="'.esc_attr(self::OPT_KEY).'['.esc_attr($key).']" value="1" '.checked(1,$val,false).'> '.esc_html($label).'</label>';
    }

    public function field_select( $args ) {
        $key   = $args['key'];
        $items = $args['items'] ?? [];
        $desc  = $args['desc'] ?? '';
        $val   = $this->opts[ $key ] ?? '';
        echo '<select name="'.esc_attr(self::OPT_KEY).'['.esc_attr($key).']">';
        foreach ( $items as $v => $label ) {
            echo '<option value="'.esc_attr($v).'" '.selected($val,$v,false).'>'.esc_html($label).'</option>';
        }
        echo '</select>';
        if ( $desc ) echo '<p class="description">'.esc_html($desc).'</p>';
    }

    public function field_text( $args ) {
        $key   = $args['key'];
        $desc  = $args['desc'] ?? '';
        $ph    = $args['placeholder'] ?? '';
        $val   = $this->opts[ $key ] ?? '';
        echo '<input type="text" class="regular-text" name="'.esc_attr(self::OPT_KEY).'['.esc_attr($key).']" value="'.esc_attr($val).'" placeholder="'.esc_attr($ph).'">';
        if ( $desc ) echo '<p class="description">'.esc_html($desc).'</p>';
    }

    public function field_textarea( $args ) {
        $key   = $args['key'];
        $desc  = $args['desc'] ?? '';
        $ph    = $args['placeholder'] ?? '';
        $val   = $this->opts[ $key ] ?? '';
        echo '<textarea class="large-text code" rows="6" name="'.esc_attr(self::OPT_KEY).'['.esc_attr($key).']" placeholder="'.esc_attr($ph).'">'.esc_textarea($val).'</textarea>';
        if ( $desc ) echo '<p class="description">'.esc_html($desc).'</p>';
    }

    public function plugin_action_links( $links ) {
        $url  = admin_url( 'options-general.php?page=indo-date-converter' );
        $link = '<a href="'.esc_url($url).'">'.esc_html__('Settings','indo-date-converter').'</a>';
        array_unshift( $links, $link );
        return $links;
    }
}

Indo_Date_Converter::instance();
