<?php
/*
Plugin Name: WhatsApp Preview OG (Auto Thumbnail)
Plugin URI: https://example.com/
Description: Otomatiskan thumbnail saat link dibagikan ke WhatsApp/Facebook dengan Open Graph. Mengambil gambar unggulan (featured image) sebagai og:image ukuran 1200x630 (full-width). Ada fallback: gambar pertama di konten atau URL default dari halaman Pengaturan.
Version: 1.0.0
Author: ChatGPT
License: GPL2+
*/

if (!defined('ABSPATH')) { exit; }

class WOP_WhatsApp_OG {
    const OPTION_KEY = 'wop_settings';
    const IMAGE_SIZE = 'wop_og_share'; // 1200x630

    public function __construct() {
        add_action('init', [$this, 'register_image_size']);
        add_action('admin_init', [$this, 'register_settings']);
        add_action('admin_menu', [$this, 'add_settings_page']);
        add_action('wp_head', [$this, 'output_open_graph_tags'], 5);
    }

    public function register_image_size() {
        // 1200x630 cocok untuk preview lebar (full-width) di WA/FB
        add_image_size(self::IMAGE_SIZE, 1200, 630, true);
    }

    public function register_settings() {
        register_setting(self::OPTION_KEY, self::OPTION_KEY, [$this, 'sanitize_settings']);
        add_settings_section('wop_main', __('Pengaturan OG untuk WhatsApp','wop'), function(){
            echo '<p>Plugin ini menambahkan tag Open Graph agar thumbnail muncul saat dibagikan ke WhatsApp/Facebook. Pastikan setiap posting memiliki Featured Image untuk hasil terbaik (1200×630).</p>';
        }, self::OPTION_KEY);

        add_settings_field('default_image', __('Gambar Default (opsional)','wop'), [$this, 'field_default_image'], self::OPTION_KEY, 'wop_main');
        add_settings_field('use_content_image', __('Pakai gambar pertama dari konten jika tidak ada Featured Image','wop'), [$this, 'field_use_content_image'], self::OPTION_KEY, 'wop_main');
    }

    public function sanitize_settings($opts) {
        $clean = [];
        $clean['default_image'] = isset($opts['default_image']) ? esc_url_raw($opts['default_image']) : '';
        $clean['use_content_image'] = !empty($opts['use_content_image']) ? 1 : 0;
        return $clean;
    }

    public function get_settings() {
        $defaults = [
            'default_image' => '',
            'use_content_image' => 1,
        ];
        return wp_parse_args(get_option(self::OPTION_KEY, []), $defaults);
    }

    public function add_settings_page() {
        add_options_page(
            'WhatsApp Preview OG',
            'WhatsApp Preview OG',
            'manage_options',
            self::OPTION_KEY,
            [$this, 'render_settings_page']
        );
    }

    public function field_default_image() {
        $opts = $this->get_settings();
        $val = esc_attr($opts['default_image']);
        echo '<input type="url" name="'.self::OPTION_KEY.'[default_image]" value="'.$val.'" class="regular-text" placeholder="https://domainmu.com/path/default-og.jpg" />';
        echo '<p class="description">Dipakai jika tidak ada Featured Image dan tidak ditemukan gambar di dalam konten. Disarankan 1200×630.</p>';
    }

    public function field_use_content_image() {
        $opts = $this->get_settings();
        $checked = $opts['use_content_image'] ? 'checked' : '';
        echo '<label><input type="checkbox" name="'.self::OPTION_KEY.'[use_content_image]" value="1" '.$checked.'> Aktif</label>';
    }

    public function render_settings_page() {
        ?>
        <div class="wrap">
            <h1>WhatsApp Preview OG</h1>
            <form method="post" action="options.php">
                <?php
                    settings_fields(self::OPTION_KEY);
                    do_settings_sections(self::OPTION_KEY);
                    submit_button();
                ?>
            </form>
            <hr />
            <h2>Tips Agar Thumbnail Melebar (Full-width)</h2>
            <ol>
                <li>Gunakan gambar minimal 1200×630px sebagai <em>Featured Image</em>.</li>
                <li>Pastikan situs memakai HTTPS dan halaman dapat diakses publik (tidak diblokir robots/meta noindex).</li>
                <li>Hindari redirection berlapis untuk URL gambar.</li>
                <li>Jika mengganti gambar, lakukan “Scrape Again” via <a href="https://developers.facebook.com/tools/debug/" target="_blank" rel="noopener">Facebook Sharing Debugger</a>.</li>
            </ol>
        </div>
        <?php
    }

    private function get_image_from_content($post) {
        if (!$post) return '';
        if (!has_filter('the_content')) {
            // ensure filters available
        }
        $content = $post->post_content;
        if (empty($content)) return '';

        // Cari <img src="...">
        if (preg_match('/<img[^>]+src=["\']([^"\']+)["\']/i', $content, $m)) {
            return esc_url_raw($m[1]);
        }
        return '';
    }

    private function get_best_image($post_id) {
        $opts = $this->get_settings();

        // 1) Featured image (prefer size 1200x630), fallback full
        if (has_post_thumbnail($post_id)) {
            $thumb_id = get_post_thumbnail_id($post_id);
            $og = wp_get_attachment_image_src($thumb_id, self::IMAGE_SIZE);
            if ($og && !empty($og[0])) {
                return ['url' => $og[0], 'width' => intval($og[1]), 'height' => intval($og[2])];
            }
            $full = wp_get_attachment_image_src($thumb_id, 'full');
            if ($full && !empty($full[0])) {
                return ['url' => $full[0], 'width' => intval($full[1]), 'height' => intval($full[2])];
            }
        }

        // 2) First image in content (if allowed)
        if (!empty($opts['use_content_image'])) {
            $img = $this->get_image_from_content(get_post($post_id));
            if ($img) {
                return ['url' => esc_url($img), 'width' => 0, 'height' => 0];
            }
        }

        // 3) Default image from settings
        if (!empty($opts['default_image'])) {
            return ['url' => esc_url($opts['default_image']), 'width' => 1200, 'height' => 630];
        }

        // 4) Site Icon (fallback)
        $icon = get_site_icon_url(512);
        if (!empty($icon)) {
            return ['url' => esc_url($icon), 'width' => 512, 'height' => 512];
        }

        return ['url' => '', 'width' => 0, 'height' => 0];
    }

    private function get_meta_description($post_id) {
        // Pakai excerpt jika ada, kalau tidak auto-generate singkat dari konten
        $post = get_post($post_id);
        if (!$post) return get_bloginfo('description');

        if (has_excerpt($post_id)) {
            return wp_strip_all_tags(get_the_excerpt($post_id));
        }
        $content = wp_strip_all_tags($post->post_content);
        $content = preg_replace('/\s+/', ' ', $content);
        $desc = mb_substr($content, 0, 160);
        return trim($desc);
    }

    public function output_open_graph_tags() {
        if (is_admin()) return;

        $url = (is_singular()) ? get_permalink() : home_url(add_query_arg([],$_SERVER['REQUEST_URI'] ?? ''));
        $site_name = get_bloginfo('name');
        $locale = get_locale();
        $type = is_singular() ? 'article' : 'website';
        $title = is_singular() ? single_post_title('', false) : get_bloginfo('name');
        $desc = is_singular() ? $this->get_meta_description(get_the_ID()) : get_bloginfo('description');
        $image = is_singular() ? $this->get_best_image(get_the_ID()) : ['url' => '', 'width'=>0, 'height'=>0];

        // Jika bukan singular (beranda/arsip), coba pakai Site Icon sebagai fallback
        if (!is_singular() && empty($image['url'])) {
            $icon = get_site_icon_url(512);
            if (!empty($icon)) {
                $image = ['url'=>$icon, 'width'=>512, 'height'=>512];
            }
        }

        // Output OG
        echo "\n<!-- WhatsApp Preview OG by WOP -->\n";
        printf("<meta property=\"og:locale\" content=\"%s\" />\n", esc_attr($locale));
        printf("<meta property=\"og:type\" content=\"%s\" />\n", esc_attr($type));
        printf("<meta property=\"og:site_name\" content=\"%s\" />\n", esc_attr($site_name));
        printf("<meta property=\"og:title\" content=\"%s\" />\n", esc_attr($title));
        printf("<meta property=\"og:description\" content=\"%s\" />\n", esc_attr($desc));
        printf("<meta property=\"og:url\" content=\"%s\" />\n", esc_url($url));

        if (!empty($image['url'])) {
            $img = esc_url($image['url']);
            printf("<meta property=\"og:image\" content=\"%s\" />\n", $img);
            printf("<meta property=\"og:image:secure_url\" content=\"%s\" />\n", $img);
            if (!empty($image['width']))  printf("<meta property=\"og:image:width\" content=\"%d\" />\n", intval($image['width']));
            if (!empty($image['height'])) printf("<meta property=\"og:image:height\" content=\"%d\" />\n", intval($image['height']));
        }

        // Tambahan: Twitter Cards (beberapa platform ikut baca)
        echo "<meta name=\"twitter:card\" content=\"summary_large_image\" />\n";
        printf("<meta name=\"twitter:title\" content=\"%s\" />\n", esc_attr($title));
        printf("<meta name=\"twitter:description\" content=\"%s\" />\n", esc_attr($desc));
        if (!empty($image['url'])) {
            printf("<meta name=\"twitter:image\" content=\"%s\" />\n", esc_url($image['url']));
        }
        echo "<!-- /WhatsApp Preview OG -->\n";
    }
}

new WOP_WhatsApp_OG();
